/***********************************************************************
* ht1632_chipselect / ht1632_chipfree
* Select or de-select a particular ht1632 chip.
* De-selecting a chip ends the commands being sent to a chip.
* CD pins are active-low; writing 0 to the pin selects the chip.
***********************************************************************/

void ht1632_chipselect(byte chipno)
{
	DEBUGPRINT("\nHT1632(%d) ", chipno);
	digitalWrite(chipno, 0);
}

void ht1632_chipfree(byte chipno)
{
	DEBUGPRINT(" [done %d]", chipno);
	digitalWrite(chipno, 1);
}

/*
* we keep a copy of the display controller contents so that we can
* know which bits are on without having to (slowly) read the device.
* Note that we only use the low four bits of the shadow ram, since
* we're shadowing 4-bit memory.  This makes things faster, and we
* use the other half for a "snapshot" when we want to plot new data
* based on older data...
*/
// (fc) covers the case for 32x8 as well (64 bytes, 4 bits)
byte ht1632_shadowram[96];  // our copy of the display's RAM

/*
* ht1632_writebits
* Write bits (up to <img src="http://timewitharduino.com/wp-includes/images/smilies/icon_cool.gif" alt="8)" class="wp-smiley"> to h1632 on pins HT1632_DATA, HT1632_WRCLK
* Chip is assumed to already be chip-selected
* Bits are shifted out from MSB to LSB, with the first bit sent
* being (bits & firstbit), shifted till firsbit is zero.
*/
void ht1632_writebits (byte bits, byte firstbit)
{
	DEBUGPRINT(" ");
	while (firstbit) {
		DEBUGPRINT((bits&firstbit ? "1" : "0"));
		digitalWrite(HT1632_WRCLK, LOW);
		if (bits & firstbit) {
			digitalWrite(HT1632_DATA, HIGH);
		}
		else {
			digitalWrite(HT1632_DATA, LOW);
		}
		digitalWrite(HT1632_WRCLK, HIGH);
		firstbit >>= 1;
	}
}

/*
* ht1632_sendcmd
* Send a command to the ht1632 chip.
* A command consists of a 3-bit "CMD" ID, an 8bit command, and
* one "don't care bit".
*   Select 1 0 0 c7 c6 c5 c4 c3 c2 c1 c0 xx Free
*/
static void ht1632_sendcmd (byte command)
{
	ht1632_chipselect(HT1632_CS);  // Select chip
	ht1632_writebits(HT1632_ID_CMD, 1<<2);  // send 3 bits of id: COMMMAND
	ht1632_writebits(command, 1<<7);  // send the actual command
	ht1632_writebits(0, 1); 	/* one extra dont-care bit in commands. */
	ht1632_chipfree(HT1632_CS); //done
}

/*
* ht1632_clear
* clear the display, and the shadow memory, and the snapshot
* memory.  This uses the "write multiple words" capability of
* the chipset by writing all 96 words of memory without raising
* the chipselect signal.
*/
void ht1632_clear()
{
	char i;
	
	ht1632_chipselect(HT1632_CS);  // Select chip
	ht1632_writebits(HT1632_ID_WR, 1<<2);  // send ID: WRITE to RAM
	ht1632_writebits(0, 1<<6); // Send address
	for (i = 0; i < 96/2; i++) // Clear entire display
		ht1632_writebits(0, 1<<7); // send 8 bits of data
	ht1632_chipfree(HT1632_CS); // done
	for (i=0; i < 96; i++)
		ht1632_shadowram[i] = 0;
}

/*
* ht1632_senddata
* send a nibble (4 bits) of data to a particular memory location of the
* ht1632.  The command has 3 bit ID, 7 bits of address, and 4 bits of data.
*    Select 1 0 1 A6 A5 A4 A3 A2 A1 A0 D0 D1 D2 D3 Free
* Note that the address is sent MSB first, while the data is sent LSB first!
* This means that somewhere a bit reversal will have to be done to get
* zero-based addressing of words and dots within words.
*/
static void ht1632_senddata (byte address, byte data)
{
	ht1632_chipselect(HT1632_CS);  // Select chip
	ht1632_writebits(HT1632_ID_WR, 1<<2);  // send ID: WRITE to RAM
	ht1632_writebits(address, 1<<6); // Send address
	ht1632_writebits(data, 1<<3); // send 4 bits of data
	ht1632_chipfree(HT1632_CS); // done
}

void ht1632_setup()
{
	pinMode(HT1632_CS, OUTPUT);
	digitalWrite(HT1632_CS, HIGH); 	// unselect (active low)
	pinMode(HT1632_WRCLK, OUTPUT);
	pinMode(HT1632_DATA, OUTPUT);
	ht1632_sendcmd(HT1632_CMD_SYSDIS);  // Disable system
	
	#ifdef _16x24_
	ht1632_sendcmd(HT1632_CMD_COMS11);  // 16*32, PMOS drivers
	#else
	// (fc)
	ht1632_sendcmd(HT1632_CMD_COMS10);  // 32x8, PMOS drivers
	#endif
	
	ht1632_sendcmd(HT1632_CMD_MSTMD); 	// Master Mode
	ht1632_sendcmd(HT1632_CMD_SYSON); 	// System on
	ht1632_sendcmd(HT1632_CMD_LEDON); 	// LEDs on
	
	for (byte i=0; i<64; i++)
		ht1632_senddata(i, 0);  // clear the display!
	
	delay(100);  // ?
}

/*
* Copy a character glyph from the myfont data structure to
* display memory, with its upper left at the given coordinate
* This is unoptimized and simply uses plot() to draw each dot.
*/
void ht1632_putchar(int x, int y, char c)
{
	// fonts defined for ascii 32 and beyond (index 0 in font array is ascii 32);
	byte charIndex;
	
	// replace undisplayable characters with blank;
	if (c < 32 || c > 126)
	{
		charIndex = 0;
	}
	else
	{
		charIndex = c - 32;
	}
	
	// move character definition, pixel by pixel, onto the display;
	// fonts are defined as one byte per row;
	for (byte row=0; row<8; row++)
	{
		byte rowDots = pgm_read_byte_near(&myfont[charIndex][row]);
		for (byte col=0; col<6; col++)
		{
			if (rowDots & (1<<(5-col)))
				plot(x+col, y+row, 1);
			else
				plot(x+col, y+row, 0);
		}
	}
}

/*
* plot a point on the display, with the upper left hand corner
* being (0,0), and the lower right hand corner being (23, 15).
* Note that Y increases going "downward" in contrast with most
* mathematical coordiate systems, but in common with many displays
* No error checking; bad things may happen if arguments are out of
* bounds!  (The ASSERTS compile to nothing by default
*/
void ht1632_plot (int x, int y, char val)
{
	if (x<0 || x>X_MAX || y<0 || y>Y_MAX)
		return;
	
	char addr, bitval;
	
	/*
	* The 4 bits in a single memory word go DOWN, with the LSB
	* (first transmitted) bit being on top.  However, writebits()
	* sends the MSB first, so we have to do a sort of bit-reversal
	* somewhere.  Here, this is done by shifting the single bit in
	* the opposite direction from what you might expect.
	*/
	bitval = 8>>(y&3);  // compute which bit will need set
	
	#ifdef _16x24_
	addr = (x<<2) + (y>>2);  // compute which memory word this is in
	#else
	// (fc)
	addr = (x<<1) + (y>>2);  // compute which memory word this is in
	#endif
	
	if (val) {  // Modify the shadow memory
		ht1632_shadowram[addr] |= bitval;
	}
	else {
		ht1632_shadowram[addr] &= ~bitval;
	}
	// Now copy the new memory value to the display
	ht1632_senddata(addr, ht1632_shadowram[addr]);
}

/*
* get_shadowram
* return the value of a pixel from the shadow ram.
*/
byte get_shadowram(byte x, byte y)
{
	byte addr, bitval;
	
	bitval = 8>>(y&3);  // compute which bit will need set
	addr = (x<<2) + (y>>2);  // compute which memory word this is in
	return (0 != (ht1632_shadowram[addr] & bitval));
}

/*
* snapshot_shadowram
* Copy the shadow ram into the snapshot ram (the upper bits)
* This gives us a separate copy so we can plot new data while
* still having a copy of the old data.  snapshotram is NOT
* updated by the plot functions (except "clear")
*/
void snapshot_shadowram()
{
	for (char i=0; i< sizeof ht1632_shadowram; i++) {
		ht1632_shadowram[i] = (ht1632_shadowram[i] & 0x0F) | ht1632_shadowram[i] << 4;  // Use the upper bits
	}
}

/*
* get_snapshotram
* get a pixel value from the snapshot ram (instead of
* the actual displayed (shadow) memory
*/
byte get_snapshotram(byte x, byte y)
{
	byte addr, bitval;
	
	bitval = 128>>(y&3);  // user upper bits!
	
	#ifdef _16x24_
	addr = (x<<2) + (y>>2);  // compute which memory word this is in
	#else
	// (fc)
	addr = (x<<1) + (y>>2);  // compute which memory word this is in
	#endif
	
	if (ht1632_shadowram[addr] & bitval)
		return 1;
	return 0;
}

/*
* This works equally well for both 16x24 and 8x32 matrices.
*/
void displayScrollingLine()
{
	// shift the whole screen 6 times, one column at a time;
	for (int x=0; x < 6; x++)
	{
		ht1632_putchar(-x, 0, msg[crtPos]);
		ht1632_putchar(-x+6,  0, ((crtPos+1 < strlen(msg)) ? msg[crtPos+1] : ' '));
		ht1632_putchar(-x+12, 0, ((crtPos+2 < strlen(msg)) ? msg[crtPos+2] : ' '));
		ht1632_putchar(-x+18, 0, ((crtPos+3 < strlen(msg)) ? msg[crtPos+3] : ' '));
		ht1632_putchar(-x+24, 0, ((crtPos+4 < strlen(msg)) ? msg[crtPos+4] : ' '));
		ht1632_putchar(-x+30, 0, ((crtPos+5 < strlen(msg)) ? msg[crtPos+5] : ' '));
		ht1632_putchar(-x+36, 0, ((crtPos+6 < strlen(msg)) ? msg[crtPos+6] : ' '));
		delay(DISPDELAY);
	}
	
	crtPos++;
	if (crtPos >= strlen(msg))
	{
		crtPos = 0;
	}
}