/***********************************************
MIDI Toy - Simple Button Test 2
Author: Jason Webb
Date: 02/29/2012
URL: http://cs.unk.edu/~webb/

Description: Tests a single button of the toy using the debouncing 
technique outlined by Limor "ladyada" Fried at:
http://www.adafruit.com/blog/2009/10/20/example-code-for-multi-button-checker-with-debouncing/

***********************************************/

// Array of all pins which are connected to buttons
byte buttons[] = {PIN_B7, PIN_D0, PIN_D1, PIN_D4, PIN_D5, PIN_D6, PIN_D7, PIN_E0, PIN_E1, PIN_C0, PIN_C1, PIN_C2, PIN_C3, PIN_B3, PIN_B2, PIN_B1, PIN_B0, PIN_E7, PIN_E6, PIN_F0, PIN_F1, PIN_F2, 32, 33, 34, 28, 29, 30, 31};

#define NUMBUTTONS sizeof(buttons)
#define DEBOUNCE 10  // button debouncer, how many ms to debounce, 5+ ms is usually plenty

volatile byte pressed[NUMBUTTONS], justpressed[NUMBUTTONS], justreleased[NUMBUTTONS];

void setup() {
  // Set up all input pins with internal pull-up
  for (byte i=0; i<NUMBUTTONS; i++)
    pinMode(buttons[i], INPUT_PULLUP);

  // Run timer2 interrupt every 15 ms 
  TCCR2A = 0;
  TCCR2B = 1<<CS22 | 1<<CS21 | 1<<CS20;

  //Timer2 Overflow Interrupt Enable
  TIMSK2 |= 1<<TOIE2;

}

void loop() {
  // Check the state of each button
  for (byte i=0; i<NUMBUTTONS; i++) {
    // Button has just been pressed
    if (justpressed[i]) {
      justpressed[i] = 0;
      Serial.print( buttons[i], DEC );
      Serial.println(" pressed");
    }
    
    // Button has just been released
    if (justreleased[i]) {
      justreleased[i] = 0;
      Serial.print( buttons[i], DEC );
      Serial.println(" released");
    }
  }
}

// Debounce code from Adafruit
SIGNAL(TIMER2_OVF_vect) {
  check_switches();
}

void check_switches()
{
  static byte previousstate[NUMBUTTONS];
  static byte currentstate[NUMBUTTONS];
  static long lasttime;
  byte index;

  if (millis() < lasttime) {
     // we wrapped around, lets just try again
     lasttime = millis();
  }

  if ((lasttime + DEBOUNCE) > millis()) {
    // not enough time has passed to debounce
    return;
  }
  // ok we have waited DEBOUNCE milliseconds, lets reset the timer
  lasttime = millis();

  for (index = 0; index < NUMBUTTONS; index++) {

    currentstate[index] = digitalRead(buttons[index]);   // read the button

    /*     
    Serial.print(index, DEC);
    Serial.print(": cstate=");
    Serial.print(currentstate[index], DEC);
    Serial.print(", pstate=");
    Serial.print(previousstate[index], DEC);
    Serial.print(", press=");
    */

    if (currentstate[index] == previousstate[index]) {
      if ((pressed[index] == LOW) && (currentstate[index] == LOW)) {
          // just pressed
          justpressed[index] = 1;
      }
      else if ((pressed[index] == HIGH) && (currentstate[index] == HIGH)) {
          // just released
          justreleased[index] = 1;
      }
      pressed[index] = !currentstate[index];  // remember, digital HIGH means NOT pressed
    }
    //Serial.println(pressed[index], DEC);
    previousstate[index] = currentstate[index];   // keep a running tally of the buttons
  }
}
