#include <Servo.h> 

// servo will be used to flip a switch on the red beacon light 
Servo myservo;  

// constants to set up pins  
int mySensors[] = {A0, A1, A2, A3, A4, A5};    // analog pins with CDS cells to detect laser beam
const int numSensors = 4;   // total number of CDS cells used ###MODIFY IF NEEDED###
const int button1 = 2;      // digital input for start button
const int button2 = 3;      // digital input for stop button
const int button3 = 4;      // digital input for reset button

// variables to handle analog values from CDS cells
int curValue;               // stores the value of the CDS cell that is currently being read
int prevValue[numSensors];  // stores the previous value of each CDS cell
int baseValue[numSensors];  // stores a baseline value for each CDS cell when the laser beam is not broken
int shouldAlarm[numSensors];// flag to decide if alarm signal should be sent
int sensitivity = 35;       // an offset from the baseValue to determine if laser beam is broken ###MODIFY IF NEEDED###
int sendAlarm = 0;          // determines if we should send the alarm signal to the COM port

// variables used to read start and stop buttons`
int state1 = HIGH;      // current state of start button
int state2 = HIGH;      // current state of stop button
int state3 = HIGH;      // current state of reset button

// variables to control red beacon light
int alarmLength = 3000; // how long red light should remain on in milliseconds - synch with mp3 played by processing ###MODIFY IF NEEDED###
unsigned long alarmTime;// time alarm was activated
int onPos = 50;          // on position for servo ###MODIFY IF NEEDED###  
int offPos = 110;        // off position for servo ###MODIFY IF NEEDED###

void setup() {
  // initialize the serial communication to PC
  Serial.begin(9600);

  // attach the servo to pin 9
  myservo.attach(9); 
  myservo.write(offPos);
  
  // set all of the CDS cell analog pins as inputs
  for (int thisSensor = 0; thisSensor < numSensors; thisSensor++)  {
    pinMode(mySensors[thisSensor], INPUT);     
  }
  
  // set all button pins as inputs
  pinMode(button1, INPUT);
  pinMode(button2, INPUT);
  pinMode(button3, INPUT); 
 
  // enable internal pull up resistors for all button pins
  digitalWrite(button1, HIGH); 
  digitalWrite(button2, HIGH); 
  digitalWrite(button3, HIGH); 

  calibrate();
}

void loop() {
  // check the push buttons to see if any are pressed
  delay(10); // debounces buttons
  int val1 = digitalRead(button1);
  int val2 = digitalRead(button2);
  int val3 = digitalRead(button3);
  if(state1 != val1 || state2 != val2 || state3 != val3) {
    // start button was pressed
    if (state1 == HIGH && val1 == LOW) {
      Serial.println("start");
    }  
    // stop button was pressed
    if (state2 == HIGH && val2 == LOW) {
      Serial.println("stop");
    }
    // reset button was pressed
    if (state3 == HIGH && val3 == LOW) {
      Serial.println("reset");
      calibrate();
    }
    state1 = val1;
    state2 = val2;
    state3 = val3;
  }

  // check each CDS cell to see if the laser beam is broken
  for (int thisSensor = 0; thisSensor < numSensors; thisSensor++)  {
    curValue = analogRead(mySensors[thisSensor]);
    
   // A broken beam is detected if the current sensor value and
   // the previous value are both less than the base value
   // minus the sensitivity. 
    if (curValue < (baseValue[thisSensor] - sensitivity) && prevValue[thisSensor] < (baseValue[thisSensor] - sensitivity) && shouldAlarm[thisSensor] == 1) {
      // laser beam has been broken, alarm should be sent
      Serial.println("alarm");
      shouldAlarm[thisSensor] = 0;
      // activate servo to turn on red beacon light
      myservo.write(onPos);
      delay(20);
      alarmTime = millis();
    }
    // laser beam not currently broken, so reset the alarm flag
    else if (curValue > (baseValue[thisSensor] - sensitivity) && prevValue[thisSensor] > (baseValue[thisSensor] - sensitivity)) {
      shouldAlarm[thisSensor] = 1;
    }
    prevValue[thisSensor] = curValue;  // update the previous value
  }

  // deactivate the servo if it has been more than alarmLength
  if (millis() - alarmTime > alarmLength) {
    myservo.write(offPos);
    delay(20);
  }
}

void calibrate() {
  // for each CDS cell, average three readings together to 
  // calculate base value when laser beam is not broken
  for (int thisSensor = 0; thisSensor < numSensors; thisSensor++)  {
    baseValue[thisSensor] = analogRead(mySensors[thisSensor]);    // first reading
    delay(20);
    baseValue[thisSensor] += analogRead(mySensors[thisSensor]);   // plus second reading
    delay(20);
    baseValue[thisSensor] += analogRead(mySensors[thisSensor]);   // plus third reading
    baseValue[thisSensor] /= 3;                                   // divided by 3 to find average
    
//    Serial.println(thisSensor);
//    Serial.println(baseValue[thisSensor]);
  }
}
