// *****************************************************************
// ****                 Automatic Tubular Bells                 ****
// ****                     www.tolaemon.com                    ****
// ****                       Source code                       ****
// ****                  01/01/2006 - 01/09/2006                ****
// ****                     Jordi Bartolome                     ****
// ****                                                         ****
// **** IMPORTANT:                                              ****
// **** Using parts of this code means accepting all conditions ****
// **** exposed on the atb web: www.tolaemon.com/atb            ****
// *****************************************************************

#include "uartparser.h"

#define UART_IN_QUEUE_MAX 16
#define COMMAND_LENGTH 13
#define UART_OUT_QUEUE_MAX 12

// variables of the UART input buffer (UART RX)
uint8_t UART_in_queue[UART_IN_QUEUE_MAX];// UARTs input buffer
uint8_t UART_in_queue_start;// UARTs input FIFOs get possition 
uint8_t UART_in_queue_end;// UARTs input FIFOs enqueue position

// variables used to store the command read from the input buffer
uint8_t command[COMMAND_LENGTH]; // array where the command is stored after being processed
uint8_t command_index;// integer used as a index when filling command array...

// variables of the UART output buffer (UART TX)
uint8_t UART_out_queue[UART_OUT_QUEUE_MAX];// UARTs ouptut buffer
uint8_t UART_out_queue_start;// ouptut FIFOs get possition 
uint8_t UART_out_queue_end;// ouptut FIFOs enqueue position


SIGNAL (SIG_USART_RECV){
	
	// Recived value is sotred into the queue buffer
	UARTPARSER_InputShift(UDR);
	
};//SIGNAL (SIG_USART_RECV)


/* Trasmision forzada, se espera a que el buffer del USART este vacio para enviar el valor  */
void USART_ForceTransmit( uint8_t data ){

	while ( !( UCSRA & (1<<UDRE)) ){
		// si el buffer de tansmision todavia no esta vacio se espera
	};//while
	// si el buffer ya esta vacio se pone el dato para q se envie
	UDR = data;
	
};//USART_Force_Transmit


/* Subroutine which intializes the USART and the buffer queue */
void UARTPARSER_Init( ){
	
	// USART is configured
	UBRRH = (uint8_t)(UART_BAUD_CALC(UART_BAUD_RATE,F_OSC)>>8);
	UBRRL = (uint8_t)UART_BAUD_CALC(UART_BAUD_RATE,F_OSC);
	UCSRA=0x20;
	UCSRB = (1 << RXEN) | (1 << TXEN) | (1 << RXCIE);
	UCSRC = (1 << URSEL) | (3 << UCSZ0);

	// input buffer queue is intialized
	UART_in_queue_start=0;
	UART_in_queue_end=0;

	// command array is intialized too
	for (command_index=0;command_index< COMMAND_LENGTH; command_index++){
	//for (command_index=0;command_index< 12; command_index++){		
		command[command_index]=' ';
	};//for
	command_index=0;

	// output buffer queue is intialized
	UART_out_queue_start=0;
	UART_out_queue_end=0;

};//UARTPARSER_Init



/* Function which recives a value and stores it in the queue. If it returns 1 it means that 
   the operation is OK, if returns 0 it means that it fails because there is not space */
uint8_t UARTPARSER_InputShift(uint8_t value){
	uint8_t return_value=1;

	// First = Last      EMPTY QUEUE
	// Last = Fitst -1   FULL QUEUE
	if ( ((UART_in_queue_end+1)%UART_IN_QUEUE_MAX)!=UART_in_queue_start) {
		UART_in_queue[UART_in_queue_end]=value;
		UART_in_queue_end=(UART_in_queue_end+1)%UART_IN_QUEUE_MAX;
	}else{
		return_value=0;
	};//if	
	return return_value;

};//UARTPARSER_Input_Shift


/* Function which retruns the value of the first element of the queue without unqueuing it. 
   It returns 1 if the operation is OK, or 0 if it fails probably because is empty  */
uint8_t UARTPARSER_InputGet(uint8_t * value){
	uint8_t return_value=1;

	// Final = Primero      COLA VACIA
	// Final = Primero -1   COLA LLENA
	if ((UART_in_queue_start)%UART_IN_QUEUE_MAX!=UART_in_queue_end){
		*value=UART_in_queue[UART_in_queue_start];
	}else{
		return_value=0;
	};//if
	
	return return_value;

};//UARTPARSER_Input_Get


/* Function which retruns the value of the first element and unqueues it. It returns
   1 if the operation is OK, or 0 if it fails probably because is empty  */
uint8_t UARTPARSER_InputUnshift(uint8_t * value){
	uint8_t return_value=1;

	// First = Last      EMPTY QUEUE
	// Last = Fitst -1   FULL QUEUE
	if ((UART_in_queue_start)%UART_IN_QUEUE_MAX!=UART_in_queue_end){
		*value=UART_in_queue[UART_in_queue_start];
		UART_in_queue_start=(UART_in_queue_start+1)%UART_IN_QUEUE_MAX;
	}else{
		return_value=0;
	};//if

	return return_value;

};//UARTPARSER_Input_Unshift


/* Function which recives a value and stores it in the ouput buffer queue ( UART TX) 
   queue. If it returns 1 it means that the operation is OK, if returns 0 it means 
   that it fails because there is not space */
uint8_t UARTPARSER_OutputShift(uint8_t value){
	uint8_t return_value=1;

	// First = Last      EMPTY QUEUE
	// Last = Fitst -1   FULL QUEUE
	if ( ((UART_out_queue_end+1)%UART_OUT_QUEUE_MAX)!=UART_out_queue_start) {
		UART_out_queue[UART_out_queue_end]=value;
		UART_out_queue_end=(UART_out_queue_end+1)%UART_OUT_QUEUE_MAX;
	}else{
		return_value=0;
	};//if	
	return return_value;

};//UARTPARSER_Output_Shift


/* Function which retruns the value of the first element int the output buffer queue 
   (UART TX) without unqueuing it. It returns 1 if the operation is OK, or 0 if it 
   fails probably because is empty  */
uint8_t UARTPARSER_OutputGet(uint8_t * value){
	uint8_t return_value=1;

	// Final = Primero      COLA VACIA
	// Final = Primero -1   COLA LLENA
	if ((UART_out_queue_start)%UART_OUT_QUEUE_MAX!=UART_out_queue_end){
		*value=UART_out_queue[UART_out_queue_start];
	}else{
		return_value=0;
	};//if
	
	return return_value;

};//UARTPARSER_Output_Get


/* Function which retruns the value of the first element in the output buffer queue 
   and unqueues it. It returns 1 if the operation is OK, or 0 if it fails probably 
   because is empty  */
uint8_t UARTPARSER_OutputUnshift(uint8_t * value){
	uint8_t return_value=1;

	// First = Last      EMPTY QUEUE
	// Last = Fitst -1   FULL QUEUE
	if ((UART_out_queue_start)%UART_OUT_QUEUE_MAX!=UART_out_queue_end){
		*value=UART_out_queue[UART_out_queue_start];
		UART_out_queue_start=(UART_out_queue_start+1)%UART_OUT_QUEUE_MAX;
	}else{
		return_value=0;
	};//if

	return return_value;

};//UARTPARSER_Output_Unshift


/* Procedure which reads the queue looking for a a command to execute, if 
   finds a command to execute it reads it and executes it. A command is 
   marked with '<' and '>' characters. */
void UARTPARSER_PollInputBuffer(){
	uint8_t read_value;
	uint8_t result;

	/* queue is read until FINAL_COMMAND_SYMBOL is found or until 
	   the end of the queue is reached, each read byte is stored
	   in the command array. If a INITIAL_COMMAND_SYMBOL is found the
	   command array content is erased (index array is set ot 0 ) */
	do{
		result=UARTPARSER_InputUnshift(&read_value);
		
		// if there was a value on the queue it is added to the command string
		if (result==1){
			// if is an initial command symbol, command string index is reset
			if (read_value==INITIAL_COMMAND_SYMBOL){
				command_index=0;
			};//if
			// byte is moved from the queue to the command string
			command[command_index]=read_value;
			// if final command symbol hasnt been read yet, command index is increased
			if (read_value!=FINAL_COMMAND_SYMBOL){
				command_index++;	
			};//if
		};//if
		
	}while ( (result==1)&&(read_value!=FINAL_COMMAND_SYMBOL) );
	
	// COMMANDS
	// START MOVEMEMENT
	//	'S' 	command mnemonic
	//	byte	number of motor movement to start 
	// CLEAR MOVEMENT
	//	'C' 	command mnemonic
	//	byte	number of motor movement to clear
	// MOVEMENTS_Add(uint8_t n_motor,uint8_t ints_cicle_level_limit,uint8_t n_cicles,uint8_t direction){
	// ADD COMPONENT TO A MOVEMENT
	//	'A'		command menmonic
	//	byte	number of motor movement to add component to
	//	byte	timer interruptions of a cicle which compound high part of duty cicle
	//	byte	number of cicles of the movement component
	//	byte	direction of the movement component
		
	/* if first byte is INITIAL_COMMAND_SYMBOL and last one is 
	   FINAL_COMMAND_SYMBOL it means that a command has been read */ 
	if ((command[0]==INITIAL_COMMAND_SYMBOL) &&	
		(command[command_index]==FINAL_COMMAND_SYMBOL) ) {
			command[0]=0;
			command[command_index]=0;
			
			// command is executed depending ont its OPCODE (first byte)
			switch (command[1]){
				case 'A'://MOVEMENTS_ADD:
					MOVEMENTS_Add(command[2],command[3],command[4],command[5]);
					UARTPARSER_OutputShift('A');
					break;	
				case 'C'://MOVEMENTS_CLEAR:
					MOVEMENTS_Clear(command[2]);
					UARTPARSER_OutputShift('C');
					break;	
				case 'S'://MOVEMENTS_START:
					MOVEMENTS_Start(command[2]);
					UARTPARSER_OutputShift('S');
					break;	
				case '1':
					MOVEMENTS_Clear(0);
					MOVEMENTS_Add(0,50,10,RIGHT_MOTION);
					MOVEMENTS_Add(0,5,1,LEFT_MOTION);
					MOVEMENTS_Start(0);
					UARTPARSER_OutputShift('1');
					break;	
				case '2':
					MOVEMENTS_Clear(2);
					MOVEMENTS_Add(2,50,10,RIGHT_MOTION);
					MOVEMENTS_Add(2,5,1,LEFT_MOTION);
					MOVEMENTS_Start(2);
					UARTPARSER_OutputShift('2');
					break;	
				case '3':
					MOVEMENTS_Clear(4);
					MOVEMENTS_Add(4,50,10,RIGHT_MOTION);
					MOVEMENTS_Add(4,5,1,LEFT_MOTION);
					MOVEMENTS_Start(4);
					UARTPARSER_OutputShift('3');
					break;	
				case '4':
					MOVEMENTS_Clear(1);
					MOVEMENTS_Add(1,50,10,RIGHT_MOTION);
					MOVEMENTS_Add(1,5,1,LEFT_MOTION);
					MOVEMENTS_Start(1);
					UARTPARSER_OutputShift('4');
					break;	
				case '5':
					MOVEMENTS_Clear(3);
					MOVEMENTS_Add(3,50,10,RIGHT_MOTION);
					MOVEMENTS_Add(3,5,1,LEFT_MOTION);
					MOVEMENTS_Start(3);
					UARTPARSER_OutputShift('5');
					break;	
				case '6':
					MOVEMENTS_Clear(5);
					MOVEMENTS_Add(5,50,10,RIGHT_MOTION);
					MOVEMENTS_Add(5,5,1,LEFT_MOTION);
					MOVEMENTS_Start(5);
					UARTPARSER_OutputShift('6');
					break;	
				case '7':
					MOVEMENTS_Clear(0);
					MOVEMENTS_Add(0,50,10,LEFT_MOTION);
					MOVEMENTS_Add(0,5,1,RIGHT_MOTION);
					MOVEMENTS_Start(0);
					UARTPARSER_OutputShift('7');
					break;	
				case '8':
					MOVEMENTS_Clear(2);
					MOVEMENTS_Add(2,50,10,LEFT_MOTION);
					MOVEMENTS_Add(2,5,1,RIGHT_MOTION);
					MOVEMENTS_Start(2);
					UARTPARSER_OutputShift('8');
					break;	
				case '9':
					MOVEMENTS_Clear(4);
					MOVEMENTS_Add(4,50,10,LEFT_MOTION);
					MOVEMENTS_Add(4,5,1,RIGHT_MOTION);
					MOVEMENTS_Start(4);
					UARTPARSER_OutputShift('9');
					break;	
				case 'X':
					MOVEMENTS_Clear(1);
					MOVEMENTS_Add(1,50,10,LEFT_MOTION);
					MOVEMENTS_Add(1,5,1,RIGHT_MOTION);
					MOVEMENTS_Start(1);
					UARTPARSER_OutputShift('X');
					break;	
				case 'Y':
					MOVEMENTS_Clear(3);
					MOVEMENTS_Add(3,50,10,LEFT_MOTION);
					MOVEMENTS_Add(3,5,1,RIGHT_MOTION);
					MOVEMENTS_Start(3);
					UARTPARSER_OutputShift('Y');
					break;	
				case 'Z':
					MOVEMENTS_Clear(5);
					MOVEMENTS_Add(5,50,10,LEFT_MOTION);
					MOVEMENTS_Add(5,5,1,RIGHT_MOTION);
					MOVEMENTS_Start(5);
					UARTPARSER_OutputShift('Z');
					break;	
				default:
					UARTPARSER_OutputShift('?');
					break;
			};//switch
	
	};//if

};// UARTPARSER_InputPollCommands


/* Procedure which reads output buffer queue looking for bytes to send through
   the UART.To avoid taking too much CPU only one caracter is sent each time */
void UARTPARSER_PollOutputBuffer(){
	uint8_t value;
	
	// if there is a byte in the output buffer queue it is sent
	if (UARTPARSER_OutputUnshift(&value)==1){
		USART_ForceTransmit(value);
	};//if

};// UARTPARSER_OutputPoll
