// *****************************************************************
// ****                 Automatic Tubular Bells                 ****
// ****                     www.tolaemon.com                    ****
// ****                       Source code                       ****
// ****                  01/01/2006 - 01/09/2006                ****
// ****                     Jordi Bartolome                     ****
// ****                                                         ****
// **** IMPORTANT:                                              ****
// **** Using parts of this code means accepting all conditions ****
// **** exposed on the atb web: www.tolaemon.com/atb            ****
// *****************************************************************

#include "movements.h"

// Procedure which initializes all movements
void MOVEMENTS_Init(){
	uint8_t i;
	
	for (i=0;i<N_MOVEMENT_COMPONENTS;i++){
		MOVEMENTS_Clear(i);
	};//for
	
};//MOVEMENTS_Init



// Procedure which starts the execution of a movement sequence. It has only one 
//parameter:
//	- n_movement: is the index of the movement which wants to be started
void MOVEMENTS_Start(uint8_t n_movement){
	
	//the movement can be started if it is in READY state or statrted again to if it is in EXECUTED state
	if ( (movement[n_movement].state==READY)||(movement[n_movement].state==FINISHED) ){
		//the exectuion index is placed to the begining of the movement components
		movement[n_movement].index=0;
		//the momvement is set to RUNNING state
		movement[n_movement].state=RUNNING;
	};//if

};//MOVEMENT_Start



// Procedure which clears a movement secuence: clears all the fields of the structure 
// array, and sets total and index counters to 0. It has only one parameter:
//	- n_movement: is the index of the movement which has to be cleared
void MOVEMENTS_Clear(uint8_t n_movement){
	uint8_t i;
	
	for (i=0;i<N_MOVEMENT_COMPONENTS;i++){
		movement[n_movement].component[i].ints_cicle_level_limit=0;
		movement[n_movement].component[i].n_cicles=0;
		movement[n_movement].component[i].direction=NO_MOTION;
		movement[n_movement].total=0;
		movement[n_movement].index=0;
		movement[n_movement].state=FREE;
	};//for

};//MOVEMENT_Check



// Function which returns 1 if the movement sequence has nothing to execute because
//it has been cleared (CLEAR state) or finished (FINISHED state). It returns 0 it the
//movement still hasn't finished.
uint8_t MOVEMENTS_Finished_or_Free(uint8_t n_motor){
	uint8_t value;
	
	if ( (movement[n_motor].state==FINISHED)||(movement[n_motor].state==FREE) ){
		value=1;
	}else{
		value=0;	
	};//if
	return (value);

};//MOVEMENTS_Finished



// Procedure which adds a new movement component to a movement sequence. It's parameters are:
//	- n_motor: is the index of the motor where the component movement has to be added
//  - ints_cicle_level_limit: "duty cilce" of the movement component added
//  - n_cicle: controls de duration of the movement component
//  - direction: specifies if the movement has to be to the right or the left
void MOVEMENTS_Add(uint8_t n_motor,uint8_t ints_cicle_level_limit,uint8_t n_cicles,uint8_t direction){
	uint8_t index;
	
	index=movement[n_motor].total;
	movement[n_motor].component[index].ints_cicle_level_limit=ints_cicle_level_limit;
	movement[n_motor].component[index].n_cicles=n_cicles;
	movement[n_motor].component[index].direction=direction;
	movement[n_motor].total++;// the total of components of the movement is increased
	movement[n_motor].state=READY;// the movement has components and is ready to execute them
	
};//MOVEMENT_Add



// Procedure which polls the state of each of the motors, looking if it has finished the
// execution of the last movement component (NO_MOTION). If motor i has executed current 
// (movement[i].index) motor movement component, the procedure initializes the corresponding
// motor with the next movement component.
void MOVEMENTS_Check(){
	uint8_t i,index;
	
	//for (i=0;i<N_MOTORS;i++){
	for (i=0;i<6;i++){
		//if movement[i] is in RUNNING state we look if last movement component has been executed. If has been executed
		//motor[i] should have finished and stay in NO_MOTION state ( if it is in RIGHT_MOTION or LEFT_MOTION state it means 
		//that it has not finished last movement component yet, so we should wait it finishes).
		if 	( (movement[i].state==RUNNING) && (MOTORS_State(i)==NO_MOTION) ) {
			// if still has movement components to execute, next movement component is executed and index is increased
			if (movement[i].index<(movement[i].total))  {
				index=movement[i].index;
				MOTORS_Set(i,movement[i].component[index].ints_cicle_level_limit,movement[i].component[index].n_cicles,movement[i].component[index].direction);
				movement[i].index++;
			}else{
				// if it has no more movement components to execute, movement is set to FINISHED state
				movement[i].state=FINISHED;
				movement[i].index=0;
			};//if
		};//if
	};//for
	MOTORS_Check();

};//MOVEMENT_Check



// Procedure which centers each motor in case it is uncentered after finishing current movement
// sequence programmed with. Centering the motor allows beating with more precission a bell next 
// time, because in this case the beat starts and ends in a known position: the center and the bell
void MOVEMENTS_Center(){
	uint8_t i,position;
	
	//for (i=0;i<N_MOTORS;i++){
	for (i=0;i<6;i++){
		// if a movement has FINISHED or is FREE we must center its bar
		if (MOVEMENTS_Finished_or_Free(i)==1){
			position=MOTORS_Position(i);
			switch (position){
				case RIGHT_POSITION:
				case BACK_POSITION:
					MOVEMENTS_Clear(i);
					MOVEMENTS_Add(i,15,1,LEFT_MOTION);
					MOVEMENTS_Start(i);
					break;
				case LEFT_POSITION:
					MOVEMENTS_Clear(i);
					MOVEMENTS_Add(i,15,1,RIGHT_MOTION);
					MOVEMENTS_Start(i);
					break;
				default:
					//MOVEMENTS_Clear(i);
					break;
			};//switch
		};//if
	};//for
	
};//MOVEMENTS_Center
