// *****************************************************************
// ****                 Automatic Tubular Bells                 ****
// ****                     www.tolaemon.com                    ****
// ****                       Source code                       ****
// ****                  01/01/2006 - 01/09/2006                ****
// ****                     Jordi Bartolome                     ****
// ****                                                         ****
// **** IMPORTANT:                                              ****
// **** Using parts of this code means accepting all conditions ****
// **** exposed on the atb web: www.tolaemon.com/atb            ****
// *****************************************************************

#include "motors.h"


//
//    flag_full_cicle     flag_full_cicle    flag_full_cicle  
//  >=================X>==================X>=================X ENERGY PULSE HAS FINISHED
//   _____              _____               _____
//  |     |            |     |             |     |
//  |     |            |     |             |     |        
//  |     |____________|     |_____________|     |__________________________________________
//
//	flag_decr_cicle
//  111111111111111111111111111111111111111111111111111111111111
//	
//  motor[i].direction
//  RIGHT                                                      NO_MOTION
//
//  ints_cicle 
//  000000000011111111110000000000111111111100000000001111111111
//  012345678901234567890123456789012345678901234567890123456789
//
//  motor[i].n_cicles
//  2                  1                   0
//
//  motor[i].ints_cicle_level_limit
//  6
//

volatile int ints_cicle=N_INTS_CICLE;
volatile unsigned char flag_full_cicle=0;
volatile unsigned char flag_decr_cicle=0;

t_motor motor[N_MOTORS];


SIGNAL (SIG_OUTPUT_COMPARE0){
	
	if (ints_cicle<=0){
		// if ints_cicle becomes 0 or lower, it means that an energy pulse cicle has finished
		// and a new energy pulse must begin flag_full_cicle is raised indacting than one cicle
		// has been executed, and ints_cicle is restarted to its inital value
		ints_cicle=N_INTS_CICLE;
		flag_full_cicle=1;
	}else{
		// if ints_cicle is not lower than 0 yet, is decreased and flag_decr_cicle is raised
		// indicating to other procedures that one clock interrupt has taken place
		ints_cicle--;
		flag_decr_cicle=1;
	};//if

};//SIGNAL (SIG_OUTPUT_COMPARE0)



// Function which returns the state of a motor, it means the direction where it is moving.
// It returns an uint8_t with the state of the n_motor. As parameters it gets:
//		- n_motor: indicating which of N_MOTORS in the array has to be consulted
uint8_t MOTORS_State(uint8_t n_motor){
	
	return motor[n_motor].direction;
	
};//MOTORS_State



// Function which returns the position of the n_motor motor
uint8_t MOTORS_Position(uint8_t n_motor){
	uint8_t value=0;

	PORTD=PORTD&0xE3;// all enables are set to 0
	// We select one enable depending which motor wants to be read
	switch (n_motor)
	{
		// if motor 0 or 1 is selected the decoder is told to activate firt latch of the multiplexer
		case 0x00:
		case 0x01:
				PORTD=PORTD|0x00;// firs enable is selected sending 00 to the decoder
				break;
		// if motor 2 or 3 is selected the decoder is told to activate second latch	of the multiplexer
		case 0x02:
		case 0x03:
				PORTD=PORTD|0x04;// second enable is selected sending 00 to the decoder
				break;
		// if motor 4 or 5 is selected the decoder is told to activate third latch	of the multiplexer
		case 0x04:
		case 0x05:
				PORTD=PORTD|0x08;// thrid enable is selected sending 00 to the decoder
				break;	
		default:
				break;
	};//switch

	// after selecting the latch we read state inputs in the microcontoller and then we 
	// erase these bits which do not belong to our motor (we do it with an & and a mask)
	// COLUMN #1 ( enable first Latch of the multiplexer)
	// Motor A ( motor 0) 	RIGHT	10 11
	// 						LEFT	01 11
	// Motor B ( motor 1) 	RIGHT	11 10
	// 					 	LEFT	11 01
	// COLUMN #2 ( enable second Latch of the multiplexer)
	// Motor A ( motor 2) 	RIGHT	10 11
	// 						LEFT	01 11
	// Motor B ( motor 3) 	RIGHT	11 10
	// 					 	LEFT	11 01
	// COLUMN #3 ( enable third Latch of the multiplexer)
	// Motor A ( motor 4) 	RIGHT	10 11
	// 						LEFT	01 11
	// Motor B ( motor 5) 	RIGHT	11 10
	// 					 	LEFT	11 01	
	value=PINA&0x0F;// 4 lower bits are read to get avobe XX XX sequence
	switch (n_motor)
	{
		// to read the state of an odd motor we have to take 2 lower bits 0..1 ( 11 XX )
		case 0x00:
		case 0x02:
		case 0x04:
			value=value&0x03;
			break;
		// to read the state of an even motor we have to read bits 2..3 ( XX 11 ) and
		// after reading them, they are shifted 2 bits in order to have them in the
		// lower position
		case 0x01:
		case 0x03:
		case 0x05:
			value=value&0x0C;				
			value=value>>2;
			break;
	};//switch

	// We set the state depending on the value read
	switch (value)
	{
		case 0x03: // 11
			value=FRONT_POSITION;
			break;
		case 0x02: // 10
			value=RIGHT_POSITION;
			break;
		case 0x01: // 01
			value=LEFT_POSITION;
			break;
		default:
			//value=BACK_POSITION;
			value=FRONT_POSITION;
			break;
	};//switch
	
	return value;

};//MOTORS_Position



// Procedure which sets output bits to activate "n_motor" in specified "direction"
void MOTORS_Set_Output( uint8_t n_motor,uint8_t direction){

	// we set the state depending on the value read
	switch (n_motor)
	{
		case 0x00:
			switch(direction)
			{
				case LEFT_MOTION:
					PORTC=PORTC|0x01;
					break;
				case RIGHT_MOTION:
					PORTC=PORTC|0x02;
					break;
				case NO_MOTION:
				case BLOCKED_MOTION:
					PORTC=PORTC&0xFC;
					break;
				default:
					break;
			};//switch
			break;
		case 0x01:
			switch(direction)
			{
				case LEFT_MOTION:
					PORTC=PORTC|0x04;
					break;
				case RIGHT_MOTION:
					PORTC=PORTC|0x08;
					break;
				case NO_MOTION:
				case BLOCKED_MOTION:
					PORTC=PORTC&0xF3;
					break;
				default:
					break;
			};//switch
			break;
		case 0x02:
			switch(direction)
			{
				case LEFT_MOTION:
					PORTC=PORTC|0x10;
					break;
				case RIGHT_MOTION:
					PORTC=PORTC|0x20;
					break;
				case NO_MOTION:
				case BLOCKED_MOTION:
					PORTC=PORTC&0xCF;
					break;
				default:
					break;
			};//switch
			break;
		case 0x03:
			switch(direction)
			{
				case LEFT_MOTION:
					PORTC=PORTC|0x40;
					break;
				case RIGHT_MOTION:
					PORTC=PORTC|0x80;
					break;
				case NO_MOTION:
				case BLOCKED_MOTION:
					PORTC=PORTC&0x3F;
					break;
				default:
					break;
			};//switch
			break;
		case 0x04:
			switch(direction)
			{
				case LEFT_MOTION:
					PORTA=PORTA|0x80;
					break;
				case RIGHT_MOTION:
					PORTA=PORTA|0x40;
					break;
				case NO_MOTION:
				case BLOCKED_MOTION:
					PORTA=PORTA&0x3F;
					break;
				default:
					break;
			};//switch
			break;
		case 0x05:
			switch(direction)
			{
				case LEFT_MOTION:
					PORTA=PORTA|0x20;
					break;
				case RIGHT_MOTION:
					PORTA=PORTA|0x10;
					break;
				case NO_MOTION:
				case BLOCKED_MOTION:
					PORTA=PORTA&0xCF;
					break;
				default:
					break;
			};//switch
			break;
		default:
			break;
	};//switch
	
};//MOTORS_Set_Output



//  Procedure which initializes the structure of the "n_motor" motor with recived parameters.
//	It's parameters are:
//	 - motor: indicating which of N_MOTORS in the array has to be modified
//	 - ints_cicle_level_limit: specifies the number of interrupts of a cycle while the 
//	 motor has to be powered ( duty-cicle)
//	 - direction: the direction where the motor has to turn 
void MOTORS_Set(uint8_t n_motor,uint8_t ints_cicle_level_limit,uint8_t n_cicles,uint8_t direction){

	motor[n_motor].n_cicles=n_cicles;
	motor[n_motor].ints_cicle_level_limit=ints_cicle_level_limit;
	motor[n_motor].direction=direction;

};//MOTORS_Set


//  Procedure which initializes all motors with it's default configuration
void MOTORS_Init(){
	uint8_t i;
	
	for (i=0;i<N_MOTORS;i++){
		MOTORS_Set(i,N_INTS_CICLE,0,NO_MOTION);
	};//for

};//MOTORS_Init


// Procedure which polls flag_full_cicle and flag_decr_cicle (raised by TIMER CTC 
// interrupt) to see if there has been any interruption which has changed the state
// of any of the motors 
void MOTORS_Check(){
	uint8_t i;

	// controls duration of whole energy pulse (successive duty cicle pulses), it means 
	// it controls how many energy pulses have to be sent to each particular motor
	if (flag_full_cicle==1)
	{				
		flag_full_cicle=0;
		// Controls the number of cicles for each motor
		for (i=0;i<N_MOTORS;i++)
		{ 
			// If "i" motor has still cicles to run its cicles are decreased 
			if (motor[i].n_cicles>0)
			{
				motor[i].n_cicles--;
				if (motor[i].n_cicles==0)
				{
					// In case a motor "i" has executed all his cicles it is marked as no motion
					motor[i].direction=NO_MOTION;
					MOTORS_Set_Output(i,NO_MOTION);
				};//if
			};//if
		};//for
	};//if
	
	// Controls de amonunt of energy of the energy pulse in each duty cicle (like a PWM)
	if (flag_decr_cicle==1)
	{
		flag_decr_cicle=0;
		// Controls de energy (duty cicle) supplied to each "i" motor
		for (i=0;i<N_MOTORS;i++)
		{
			// First looks if the motor is in motion or not
			if (motor[i].direction!=NO_MOTION)
			{
				// If the motor is in motion looks on which part of the duty cicle it is
				if (ints_cicle<motor[i].ints_cicle_level_limit)
				{
					// While is in the Active part of the pulse is excited on indicated direction
					MOTORS_Set_Output( i,motor[i].direction);
				}else{
					// While is in the INactive part of the pulse it is blocked, it means no excited
					MOTORS_Set_Output( i,BLOCKED_MOTION);
				};//if
			};//if
		};//for
	};//if

};//MOTORS_CheckMovement
