/*
Keypad sketch - prints the key pressed on a keypad to the serial port
                uses the internal resistors to pull-up the rows

Outpus        - if password ok, enables the servo on pin11 and lights led on pin13 2 seconds
              - if password wrong blinks 5 times led on pin13
*/

#include <Password.h>
#include <Servo.h>

Servo myservo;                                             // declares servo
Password password = Password( "4141" );                    // password to unlock box, can be changed

const int numRows = 4;                                     // number of rows in the keypad
const int numCols = 4;                                     // number of columns
const char keymap[numRows][numCols] = {                    // Define the Keymap
    { '1', '2', '3', 'A' } ,
    { '4', '5', '6', 'B' } ,
    { '7', '8', '9', 'C' } ,
    { '*', '0', '#', 'D' }
};

// RMM this pin combination gives me the correct configuration for the plasma 16 4x4 keypad
// (see photos for new wiring using internal resistors 20130425)
// keypad seen from front:
//   pin1 = row 1 = arduino pin2
//   pin2 = row 2 = arduino pin3
//   pin3 = row 3 = arduino pin4
//   pin4 = row 4 = arduino pin5
//   pin5 = col 5 = arduino pin6
//   pin6 = col 6 = arduino pin7
//   pin7 = col 7 = arduino pin8
//   pin8 = col 8 = arduino pin9

const int rowPins[numRows] = { 2, 3, 4, 5 };               // Connect keypad ROW0, ROW1, ROW2 and ROW3 to these Arduino pins.
const int colPins[numCols] = { 6, 7, 8, 9 };               // Connect keypad COL0, COL1, COL2 and COL3 to these Arduino pins.

const int debounceTime = 20;                               // number of milliseconds for switch to be stable

void setup() {
    Serial.begin(9600);
    delay(200); 

    pinMode(13, OUTPUT);                                   // green light - light 1 second if ok
    pinMode(12, OUTPUT);                                   // red light   - blink 5 times led on pin13 if wrong 
    myservo.attach(11);                                    // servo on digital pin 11
    myservo.write(0);                                      // lock

    for (int row = 0; row < numRows; row++) {
        pinMode(rowPins[row],INPUT);                       // Set row pins as input
        digitalWrite(rowPins[row],HIGH);                   // turn on Pull-ups
    }
    for (int column = 0; column < numCols; column++) {
        pinMode(colPins[column],OUTPUT);                   // Set column pins as outputs for writing
        digitalWrite(colPins[column],HIGH);                // Make all columns inactive
    }
}

void loop() {
    char key = getKey();
    if( key != 0) {                                        // if the character is not 0 then it's a valid key press
        Serial.print("Got key ");
        Serial.println(key);
        delay(10);
	    
        switch (key) {
            case 'A': checkPassword(); delay(1); break;    // Enter password
            case 'B': password.reset(); delay(1); break;   // Clear buffer
            default: password.append(key); delay(1);       // add key to password
        }
    }
}

void checkPassword() {  
    if (password.evaluate()) {                             // if password is right open box
        Serial.println("Accepted");
        delay(10);   
        password.reset(); delay(1);
        myservo.write(90);                                 // unlock

        digitalWrite(13, HIGH);                            // turn green light on
        delay(2000);                                       // wait 2 seconds
        digitalWrite(13, LOW);                             // turn off

        myservo.write(0);                                  // lock
    } else {
        Serial.println("Denied");                          // if passwords wrong keep box locked
        delay(10);
        password.reset(); delay(1);
        myservo.write(0);                                  // lock

        for (int t = 0; t < 5; t++) {
            digitalWrite(13, HIGH);                        // turn red light on
            delay(80);                                     // wait .5 seconds
            digitalWrite(13, LOW);                         // turn off
            delay(80);                                     // wait .5 seconds
        }
    }
}
// returns with the key pressed, or 0 if no key is pressed
char getKey() {
    char key = 0;                                          // 0 indicates no key pressed
    for(int column = 0; column < numCols; column++) {
        digitalWrite(colPins[column],LOW);                 // Activate the current column.
        for(int row = 0; row < numRows; row++) {           // Scan all rows for a key press.
            if(digitalRead(rowPins[row]) == LOW) {         // Is a key pressed?
                delay(debounceTime);                       // debounce
                while(digitalRead(rowPins[row]) == LOW) ;  // wait for key to be released
                key = keymap[row][column];                 // Remember which key was pressed.
            }
        }
        digitalWrite(colPins[column],HIGH);                // De-activate the current column.
    }
    return key;                                            // returns the key pressed or 0 if none
}
